// Copyright (c) 2013 Analog Devices, Inc.  All rights reserved.

// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
  // - Redistributions of source code must retain the above copyright notice,
  // this list of conditions and the following disclaimer.
  // - Redistributions in binary form must reproduce the above copyright notice,
  // this list of conditions and the following disclaimer in the documentation 
  // and/or other materials provided with the distribution.  
  // - Modified versions of the software must be conspicuously marked as such.
  // - This software is licensed solely and exclusively for use with 
  // processors/products manufactured by or for Analog Devices, Inc.
  // - This software may not be combined or merged with other code in any manner 
  // that would cause the software to become subject to terms and conditions which 
  // differ from those listed here.
  // - Neither the name of Analog Devices, Inc. nor the names of its contributors
  // may be used to endorse or promote products derived from this software without 
  // specific prior written permission.
  // - The use of this software may or may not infringe the patent rights of one
  // or more patent holders.  This license does not release you from the requirement
  // that you obtain separate licenses from these patent holders to use this software.

// THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES, INC. AND CONTRIBUTORS "AS IS" 
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
// NON-INFRINGEMENT, TITLE, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
// ARE DISCLAIMED. IN NO EVENT SHALL ANALOG DEVICES, INC. OR CONTRIBUTORS BE LIABLE
 // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE OR 
 // CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, DAMAGES ARISING OUT OF 
 // CLAIMS OF INTELLECTUAL PROPERTY RIGHTS INFRINGEMENT; PROCUREMENT OF SUBSTITUTE
 // GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 // HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 // LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 // OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// 20180927-7CBSD SLA

`timescale 1ns / 1ps
//////////////////////////////////////////////////////////////////////////////////
// Company: 
// Engineer: 
// 
// Create Date:    17:40:43 09/23/2013 
// Design Name: 
// Module Name:    timing_gen 
// Project Name: 
// Target Devices: 
// Tool versions: 
// Description: 
// Generate ADC timing as datasheet requirement
// Dependencies: 
//
// Revision: 
// Revision 0.01 - File Created
// Additional Comments: 
//
//////////////////////////////////////////////////////////////////////////////////
module sar_timing_gen(
    io_clk,		//200Mhz
    sys_clk,	//100Mhz
    reset,
    sample_trigger,
    cnv_period,
    num_samples,
    burst_clk,
	 burst_clk_delay,
    cnv_out,
	 busy,
	 sdi_out,
	 mode_select
    );
	input			sys_clk;
	input			io_clk;
	input			reset;
	input			sample_trigger;
	input [15:0]		cnv_period;
	input [3:0]	num_samples;
	output			burst_clk;
	output			burst_clk_delay;
	output			cnv_out;
	output			busy;
	output			sdi_out;
	input	[2:0] 	mode_select;

	reg [2:0] mode;

	reg				flag = 1'b0;
	reg				flag_toggle = 1'b0;
	reg [2:0]		flag_sync = 3'b0;
	wire			flag_trigger;
	
	wire [15:0]		clk_counter;
	reg	[15:0]		cnv_period_reg = 16'b0;
	
	reg				trigger_toggle = 1'b0;
	reg [2:0]		trigger_sync = 3'b0;
	wire				trigger;
	reg [2:0]		trig_gen = 3'b0;
	
	reg	[7:0]		cnv_pulse_count = 8'd0;
	reg	[7:0]		cnv_pulse_count1 = 8'd0;
	reg				cnv_reg = 1'b0;
	reg				sdi_reg = 1'b1;
	reg 				clk_reg = 1'b0;
	reg				burst_delay = 1'b0;
	reg				burst_delay1 = 1'b0;
	reg				burst_delay2 = 1'b0;
	reg				burst_delay3 = 1'b0;

	
	reg				clk_gate = 1'b0;
	
	reg [24:0]		num_samples_reg = 25'b0;
	reg	[24:0]	num_samples_temp = 25'b0;
	reg				sampling = 1'b0;
	reg				sampling_temp = 1'b0;
	reg				first_sample = 1'b0;
	reg				reload = 1'b0;


	reg	[2:0]		flag_delay = 3'b0;
	assign busy = sampling;
	reg count_load = 1'b1;
	
	assign cnv_out =  ((first_sample) || (sampling)) ? cnv_reg : 1'b0;
	assign sdi_out = ((first_sample) || (sampling)) ? sdi_reg : 1'b1;
	assign burst_clk = (sampling) ? burst_delay2 : 1'b0;//burst_clk is 100Mhz
	assign burst_clk_delay = (sampling) ? burst_delay: 1'b0;//burst_clk is 100Mhz
//	BUFGCE (.I (burst_delay), .CE((first_sample) || (sampling)), .O(burst_clk_delay));

	reg		[7:0] trigger_cnt;//self clean count


// generates a toggle signal from the trigger pulse
	always @ (posedge sys_clk) begin
		if(reset)begin
			trigger_toggle <= 1'b0;
		end
		else begin
			if(sample_trigger)begin
				trigger_toggle <= 1'b1;
			end
			else if(trigger_cnt == 8'hf0)begin
				trigger_toggle <= 1'b0;
			end
		end
	end
	
	always @ (posedge sys_clk) begin
		if(reset)begin
			trigger_cnt <= 8'b0;
		end
		else begin
			if(trigger_toggle)begin
				trigger_cnt <= trigger_cnt + 1'b1;
			end
			else if(trigger_toggle == 1'b0)begin
				trigger_cnt <= 8'b0;
			end
		end
	end
	

// resync toggle trigger into io_clk domain
	always @ (negedge io_clk) begin
		if(reset)begin
			trigger_sync <= 3'b0;
		end
		else begin
			trigger_sync <= {trigger_sync[1:0], trigger_toggle};		
		end
	end

	// creates trigger pulse sync'd to io clk
	assign trigger = ((trigger_sync[2] ^ trigger_sync[1]) && trigger_sync[1]);////rising edge detect
//----------------------------------------------------------------------
//	loads the cnv rate value into register
	always @ (negedge io_clk) begin
		if (trigger) begin
			cnv_period_reg <= cnv_period - 1'b1;//
			mode <= mode_select;
		end
	end

// generate the counter clk
	always @ (negedge io_clk) begin
		if (clk_counter == 16'd2) begin	// allow for latency in IP counter
			count_load <= 1'b1;
		end
		else begin
			count_load <= 1'b0;
		end
	end
	
// Primitive/IP block required from Xilinx to run @ 200MHz
c_counter_binary_v11_0 i_cnv_counter(
	.clk	(~io_clk),
	.load	(count_load),
	.l		(cnv_period_reg),	// 15:0
	.q		(clk_counter)		// 15:0
);


// pulse flag to sync other counters on this clock domain
	always @ (negedge io_clk) begin
		if (clk_counter == 16'd0)
			flag <= 1'b1;
		else
			flag <= 1'b0;
	end
	
//----------------------------------------------------------------------
//counter number = 200Mhz/SPS
//mode0-----3wire-read-after-cnv-max-1.8M  
//mode1-----3wire-read-during-cnv-max-2M
//mode4-----3wire-read-****after-cnv-status****--max-1.5M  
//mode5-----3wire-read-****during-cnv-status****--max-1.8M 
// counter for cnv pulse width
	always @ (negedge io_clk) begin
	case (mode)
		3'd0: begin
          if (flag) begin
               cnv_pulse_count <= 8'd105;//2 latency+ 4cycle delay 111-6= 105
          end
          else if (cnv_pulse_count != 8'd0) begin
               cnv_pulse_count <= cnv_pulse_count - 1'b1;
          end
		end
		3'd1: begin
          if (flag) begin
               cnv_pulse_count <= 8'd94;
          end
          else if (cnv_pulse_count != 8'd0) begin
               cnv_pulse_count <= cnv_pulse_count - 1'b1;
          end
		end
		3'd4: begin
          if (flag) begin
               cnv_pulse_count <= 8'd127;
          end
          else if (cnv_pulse_count != 8'd0) begin
               cnv_pulse_count <= cnv_pulse_count - 1'b1;
          end
		end
		3'd5: begin
          if (flag) begin
               cnv_pulse_count <= 8'd105;
          end
          else if (cnv_pulse_count != 8'd0) begin
               cnv_pulse_count <= cnv_pulse_count - 1'b1;
          end
		end		
		default:begin
		
		end
	endcase
	end

// generates sdi pulse
	always @ (negedge io_clk) begin
	case (mode)	
		3'd0: begin
			sdi_reg <= 1'b1;
		end
		3'd1: begin
			sdi_reg <= 1'b1;
		end		
		3'd4: begin
			sdi_reg <= 1'b1;
		end
		3'd5: begin
			sdi_reg <= 1'b1;
		end			
		default:begin
		
		end
	endcase	
	end
	
// generates cnv pulse
	always @ (negedge io_clk) begin
	case (mode)
		3'd0: begin
          if (cnv_pulse_count >= 8'd45) begin//tconv = 300ns
               cnv_reg <= 1'b1;
          end
          else begin
               cnv_reg <= 1'b0;
          end
		end
		3'd1: begin
          if (cnv_pulse_count >= 8'd57) begin//tquiet1 = 190ns
               cnv_reg <= 1'b1;
          end
          else begin
               cnv_reg <= 1'b0;
          end
		end		
		3'd4: begin
          if (cnv_pulse_count >= 8'd67) begin
               cnv_reg <= 1'b1;
          end
          else begin
               cnv_reg <= 1'b0;
          end
		end
		3'd5: begin
          if (cnv_pulse_count >= 8'd65) begin
               cnv_reg <= 1'b1;
          end
          else begin
               cnv_reg <= 1'b0;
          end
		end		
		default:begin
		
		end
	endcase
	end

//generate for clk_reg
	always @ (negedge io_clk) begin
	case (mode)
		3'd0: begin
          if ((cnv_pulse_count >= 8'd9) && (cnv_pulse_count < 8'd45)) begin//18bits
               clk_reg <= ~clk_reg;
          end
			 else begin
					clk_reg <= 1'b0;
			 end
		end
		3'd1: begin
          if ((cnv_pulse_count >= 8'd21) && (cnv_pulse_count < 8'd57)) begin//18bits
               clk_reg <= ~clk_reg;
          end
			 else begin
					clk_reg <= 1'b0;
			 end
		end		
		3'd4: begin
          if ((cnv_pulse_count >= 8'd19) && (cnv_pulse_count < 8'd67)) begin//24bits
               clk_reg <= ~clk_reg;
          end
			 else begin
					clk_reg <= 1'b0;
			 end
		end
		3'd5: begin
          if ((cnv_pulse_count >= 8'd17) && (cnv_pulse_count < 8'd65)) begin//24bits
               clk_reg <= ~clk_reg;
          end
			 else begin
					clk_reg <= 1'b0;
			 end
		end		
		default:begin
		
		end
	endcase
	end
/*******************************delay 1cycle*******************************************/
	always @ (negedge io_clk) begin
		if (reset)begin
			burst_delay1 <= 1'b0;
			burst_delay2 <= 1'b0;
			burst_delay3 <= 1'b0;
			burst_delay <= 1'b0;
		end
		else begin
			burst_delay1 <= clk_reg;
			burst_delay2 <= burst_delay1;
			burst_delay3 <= burst_delay2;
			burst_delay <= burst_delay3;
		end
	end

//----------------------------------------------------------------------
	always @ (negedge io_clk) begin
		if (cnv_pulse_count != 3'b0) begin
			clk_gate <= 1'b1;
		end
		else begin
			clk_gate <= 1'b0;
		end
	end	
//----------------------------------------------------------------------

// resync clk_gate into sys_clk domain as an indicator of end of clk burst
	always @ (posedge sys_clk) begin
		if (reset)
			flag_sync <= 3'b0;
		else
			flag_sync <= {flag_sync[1:0], clk_gate};
	end

// creates trigger pulse sync'd to sys_clk on falling edge of clk_gate
	assign flag_trigger = ((flag_sync[2] ^ flag_sync[1]) && (flag_sync[2]));
//----------------------------------------------------------------------
//	loads the number of samples value into temp register
	always @ (posedge sys_clk) begin
		if (reset) begin
			num_samples_temp <= 25'b0;
			reload <= 1'b0;
		end
		else if (sample_trigger) begin
			num_samples_temp <= (25'h1 << (num_samples + 8'd8));// + 1'b1;
			reload <= 1'b1;
		end
		else if ((reload) && (flag_trigger)) begin // only load at end of burst
			reload <= 1'b0;								 
		end
	end
	
	always @ (posedge sys_clk) begin
		if (reset) begin
			num_samples_reg <= 25'b0;
			first_sample <= 1'b0;
		end
		else if ((reload) && (flag_trigger)) begin
			num_samples_reg <= num_samples_temp + 25'b1;
			//first_sample <= 1'b1;
		end
		else if ((flag_trigger) && (num_samples_reg > 25'b0)) begin
			num_samples_reg <= num_samples_reg - 1'b1;	
			first_sample <= 1'b1;	
		end
		else if((flag_trigger) && (num_samples_reg <= num_samples_temp))begin
			first_sample <= 1'b0;
		end

	end
		
	always @ (posedge sys_clk) begin
		if ((num_samples_reg > 25'b0)&&(num_samples_reg <= num_samples_temp))begin
			sampling_temp <= 1'b1;
		end
		else begin
			sampling_temp <= 1'b0;
		end
	end

	always @ (posedge sys_clk) begin
		if (reset) begin
			sampling <= 1'b0;
		end
		if (flag_trigger) begin
			sampling <= sampling_temp;
		end
	end


endmodule
